<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Branch;
use App\Models\Catatan;
use App\Models\Poligon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;
use RealRashid\SweetAlert\Facades\Alert;

class PoligonController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $catatans = Catatan::with('user')->orderBy('created_at', 'asc')->get();
        $user = User::findOrFail(auth()->user()->id);
        $query = Poligon::with('branch');

        // Filter berdasarkan branch
        if ($request->has('branch')) {
            $query->where('id_branch', $request->branch);
        }

        // Filter berdasarkan status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        //filter berdasarkan akun
        if ($user->hak_akses === 'Mitra'){
            $query->where('id_user', $user->id);
        }

        $data = $query->get();
        return view('order.poligon', compact('data', 'catatans'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $branch = Branch::get();
        return view('order.resource.poligon.create', compact('branch'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validasi = $request->validate([
            'branch'    => 'required',
            'nama'      => 'required',
            'tipe'      => 'required',
            'file_kml'  => 'required|max:4096',
            'file_boq'  => 'required|mimes:xlsx|max:4096',
        ],[
            'branch.required'       => 'Branch Harus Diisi!',
            'nama.required'         => 'Nomor Surat Harus Diisi!',
            'tipe.required'         => 'Perihal Surat Harus Diisi!',
            'file_kml.required'     => 'File KML Harus Diisi!',
            'file_boq.required'     => 'File BoQ Harus Diisi!',
            'file_boq.mimes'        => 'File BoQ Harus Ekstensi XLSX',
        ]);
        
        DB::beginTransaction();
        try {
            $kml = $request->file('file_kml');
            $boq = $request->file('file_boq');

            // Ambil nama file asli (atau bisa kamu custom jika perlu)
            $kmlName = $kml->getClientOriginalName();
            $boqName = $boq->getClientOriginalName();

            // Simpan ke storage/app/public/file/kml dan file/boq
            $userId = auth()->user()->id;
            $pathkml = $kml->storeAs("file/kml/{$userId}", time() . '_' . $kmlName, 'local');
            $pathboq = $boq->storeAs("file/boq/{$userId}", time() . '_' . $boqName, 'local');
            $data = array_map('strip_tags', [
                'id_branch'     => $validasi['branch'],
                'nama'          => $validasi['nama'],
                'tipe'          => $validasi['tipe'],
                'file_kml'      => $pathkml,
                'file_boq'      => $pathboq,
                'id_user'       => $userId,
            ]);
            // Baca file Excel hanya untuk 2 sel tanpa load semua baris
            $spreadsheet = IOFactory::load($boq->getPathname());
            $worksheet = $spreadsheet->getActiveSheet();

            // Ambil nilai langsung dari sel G285 dan G73
            $rawBoq = $worksheet->getCell('G285')->getCalculatedValue();
            $rawPort = $worksheet->getCell('G73')->getCalculatedValue();

            // Bersihkan dari karakter non-digit
            $cleanBoq = preg_replace('/\D/', '', $rawBoq);
            $cleanPort = preg_replace('/\D/', '', $rawPort);

            // Fungsi untuk ambil digit dinamis berdasarkan panjang angka
            function ambilDigitDinamyc($angka) {
                $panjang = strlen($angka);
                if ($panjang >= 7) {
                    return (int) substr($angka, 0, 6);
                } elseif ($panjang == 6) {
                    return (int) substr($angka, 0, 5);
                } elseif ($panjang == 5) {
                    return (int) substr($angka, 0, 4);
                } else {
                    // Jika kurang dari 5 digit, ambil semua angka
                    return (int) $angka;
                }
            }

            $totalBoq = is_numeric($cleanBoq) ? ambilDigitDinamyc($cleanBoq) : 0;
            $totalPort = is_numeric($cleanPort) ? (int) $cleanPort : 0; // totalPort tetap diambil semua angka


            $data['total_boq'] = $totalBoq;
            $data['total_port'] = $totalPort;
            $simpanData = Poligon::create($data);
            Catatan::create([
                'id_user' => $userId,
                'catatan' => 'Data dan file sudah di upload',
                'id_poligon' => $simpanData->id,
            ]);
            DB::commit();
            Alert::success('Data Berhasil Di Tambahkan!');
            return redirect()->route('poligon.index');
        } catch (\Exception $e) {
            DB::rollback(); //batalkan transaksi jika ada error
            Alert::error('Terjadi kesalahan' . $e->getMessage());
            return back();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $branch = Branch::get();
        $data = Poligon::findOrFail($id);
        $user = auth()->user();

        $allowedRoles = ['Admin', 'SDI'];

        $isAdminOrSdi = false;
        if (is_string($user->hak_akses)) {
            $isAdminOrSdi = in_array($user->hak_akses, $allowedRoles);
        } elseif (is_array($user->hak_akses)) {
            $isAdminOrSdi = count(array_intersect($allowedRoles, $user->hak_akses)) > 0;
        }
        if (!$isAdminOrSdi && $data->id_user != $user->id) {
            abort(403, 'Tidak diizinkan mengakses poligon ini');
        }
        return view('order.resource.poligon.show', compact('data', 'branch'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $branch = Branch::get();
        $data = Poligon::findOrFail($id);
        $user = auth()->user();

        $allowedRoles = ['Admin', 'SDI'];

        $isAdminOrSdi = false;
        if (is_string($user->hak_akses)) {
            $isAdminOrSdi = in_array($user->hak_akses, $allowedRoles);
        } elseif (is_array($user->hak_akses)) {
            $isAdminOrSdi = count(array_intersect($allowedRoles, $user->hak_akses)) > 0;
        }
        if (!$isAdminOrSdi && $data->id_user != $user->id) {
            abort(403, 'Tidak diizinkan mengakses poligon ini');
        }
        return view('order.resource.poligon.edit', compact('data', 'branch'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validasi = $request->validate([
            'branch'        => 'required',
            'nama'          => 'required',
            'tipe'          => 'required',
            'file_kml'      => 'nullable|max:4096',
            'file_boq'      => 'nullable|mimes:xlsx|max:4096'
        ],[
            'branch.required'       => 'Branch harus dipilih!',
            'nama.required'         => 'Nama harus diisi!',
            'tipe'                  => 'Tipe harus dipilih!',
            'file_kml.max'          => 'File KML Max ukuran 4MB',
            'file_boq.mimes'        => 'File BoQ harus ekstensi XLSX',
            'file_boq.max'          => 'File BoQ Max ukuran 4MB',
        ]);
        DB::beginTransaction();
        try {
            $poligon = Poligon::findOrFail($id);
            $catatan = new Catatan();
            $user = auth()->user();
            // Inisialisasi kosong
            $catatanFile = [];
            $data = array_map('strip_tags', [
                'id_branch'     => $validasi['branch'],
                'nama'          => $validasi['nama'],
                'tipe'          => $validasi['tipe'],
            ]);
            if ($request->hasFile('file_kml')) {
                $kml = $request->file('file_kml');
                $kmlName = time() . '_' . $kml->getClientOriginalName();
                

                // Hapus file lama jika ada
                if ($poligon->file_kml && Storage::exists($poligon->file_kml)) {
                    Storage::delete($poligon->file_kml);
                }
                // Simpan File kml baru
                $pathkml = $kml->storeAs("file/kml/{$poligon->id_user}", $kmlName, 'local');
                $data['file_kml'] = $pathkml; // Simpan path ke database
                $catatanFile[] = 'KML';
            }
            if ($request->hasFile('file_boq')) {
                $boq = $request->file('file_boq');
                $boqName = time() . '_' . $boq->getClientOriginalName();
                

                // Hapus file lama jika ada
                if ($poligon->file_boq && Storage::exists($poligon->file_boq)) {
                    Storage::delete($poligon->file_boq);
                }
                // Simpan foto baru
                $pathboq = $boq->storeAs("file/boq/{$poligon->id_user}", $boqName, 'local');
                if (!$pathboq) {
                    dd('Gagal simpan file!');
                }
                // Baca file Excel hanya untuk 2 sel tanpa load semua baris
                $spreadsheet = IOFactory::load($boq->getPathname());
                $worksheet = $spreadsheet->getActiveSheet();

                // Ambil nilai langsung dari sel G285 dan G73
                $rawBoq = $worksheet->getCell('G285')->getCalculatedValue();
                $rawPort = $worksheet->getCell('G73')->getCalculatedValue();

                // Bersihkan dari karakter non-digit
                $cleanBoq = preg_replace('/\D/', '', $rawBoq);
                $cleanPort = preg_replace('/\D/', '', $rawPort);

                // Fungsi untuk ambil digit dinamis berdasarkan panjang angka
                function ambilDigitDinamyc($angka) {
                    $panjang = strlen($angka);
                    if ($panjang >= 7) {
                        return (int) substr($angka, 0, 6);
                    } elseif ($panjang == 6) {
                        return (int) substr($angka, 0, 5);
                    } elseif ($panjang == 5) {
                        return (int) substr($angka, 0, 4);
                    } else {
                        // Jika kurang dari 5 digit, ambil semua angka
                        return (int) $angka;
                    }
                }

                $totalBoq = is_numeric($cleanBoq) ? ambilDigitDinamyc($cleanBoq) : 0;
                $totalPort = is_numeric($cleanPort) ? (int) $cleanPort : 0; // totalPort tetap diambil semua angka
                $data['file_boq'] = $pathboq; // Simpan path ke database
                $catatanFile[] = 'BoQ';
                $data['total_boq'] = $totalBoq;
                $data['total_port'] = $totalPort;
            }
            $data['status'] = 'review';
            $poligon->update($data);

            // Update catatan dengan deskripsi file yang ada
            $catatanText = 'Data ' . $data['nama'] . ' berhasil diperbarui';

            if (!empty($catatanFile)) {
                $catatanText .= ' dan file ' . implode(' & ', $catatanFile) . ' diupdate';
            }
            $catatan->id_user = $user->id;
            $catatan->catatan = $catatanText;
            $catatan->id_poligon = $poligon->id;
            $catatan->save();
            DB::commit();
            return redirect()->route('poligon.index')->with('success', 'Data Berhasil Di Update');
        } catch (\Exception $e) {
            DB::rollback(); //batalkan transaksi jika ada error
            Alert::error('Terjadi kesalahan' . $e->getMessage());
            return back();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $data = Poligon::findOrFail($id);

        // Hapus file jika ada
        if ($data->file_boq && Storage::exists($data->file_boq)) {
            Storage::delete($data->file_boq);
        }

        if ($data->file_kml && Storage::exists($data->file_kml)) {
            Storage::delete($data->file_kml);
        }

        // Hapus semua catatan terkait
        Catatan::where('id_poligon', $data->id)->get()->each->delete();

        // Hapus data poligon
        $data->delete();

        return redirect()->route('poligon.index')->with('success', 'Data Berhasil dihapus');
    }


    public function updateStatus(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:poligons,id',
            'status' => 'required|in:review,revisi,approval,kendala',
        ]);

        $poligon = Poligon::find($request->id);
        $poligon->status = $request->status;
        $poligon->save();

        return response()->json(['message' => 'Status berhasil diupdate.']);
    }
}
