<?php

namespace App\Http\Controllers;

use App\Models\OTP;
use App\Models\User;
use App\Models\Catatan;
use App\Models\Poligon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class PenggunaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $data = User::get();
        return view('main.pengguna', compact('data'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('main.resource.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validasi = $request->validate([
            'nama'          => 'required|string|max:255',
            'nomor_induk'   => 'required|string|unique:users,nomor_induk',
            'email'         => 'nullable|email|unique:users,email',
            'password'      => 'required|string|min:8',
            'hak_akses'     => 'required|in:Admin,Mitra,SDI',
        ],[
            'nama.required'             => 'Nama harus diisi!',
            'nomor_induk.required'      => 'Nomor induk harus diisi!',
            'password.required'         => 'Password harus diisi',
            'hak_akses.required'        => 'Hak akses harus dipilih!',
            'email.unique'              => 'Email tidak boleh sama dengan yang terdaftar!',
            'password.min'              => 'Password minimal 8 karakter!',
            'nomor_induk'               => 'Nomor induk tidak boleh sama dengan yang terdaftar!',
        ]);

        $data = array_map('strip_tags', [
                'nama'          => $validasi['nama'],
                'nomor_induk'   => $validasi['nomor_induk'],
                'hak_akses'     => $validasi['hak_akses'],
                'email'         => $validasi['email'] ?? NULL,
            ]);

        // Simpan data user baru
        User::create([
            'nama'          => $data['nama'],
            'nomor_induk'   => $data['nomor_induk'],
            'email'         => $data['email'],
            'password'      => bcrypt(strip_tags($validasi['password'])), // Enkripsi password
            'hak_akses'     => $data['hak_akses'],
        ]);

        return redirect()->route('pengguna.index')->with('success', 'Data pengguna berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $data = User::findOrFail($id);
        return view('main.resource.edit', compact('data'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validasi = $request->validate([
            'nama'          => 'required|string|max:255',
            'nomor_induk'   => 'required|string|unique:users,nomor_induk,' . $id,
            'email'         => 'nullable|email|unique:users,email,' . $id,
            'password'      => 'nullable|string|min:8',
            'hak_akses'     => 'required|in:Admin,Mitra,SDI',
        ],[
            'nama.required'             => 'Nama harus diisi!',
            'nomor_induk.required'      => 'Nomor induk harus diisi!',
            'hak_akses.required'        => 'Hak akses harus dipilih!',
            'email.unique'              => 'Email tidak boleh sama dengan yang terdaftar!',
            'password.min'              => 'Password minimal 8 karakter!',
            'nomor_induk'               => 'Nomor induk tidak boleh sama dengan yang terdaftar!',
        ]);
        DB::beginTransaction();
        try {
            $data = array_map('strip_tags', [
                'nama'          => $validasi['nama'],
                'nomor_induk'   => $validasi['nomor_induk'],
                'hak_akses'     => $validasi['hak_akses'],
            ]);
            if (!empty($validasi['password'])) {
                $data['password'] = bcrypt(strip_tags($validasi['password']));
            }
            if (!empty($validasi['email'])) {
                $data['email'] = strip_tags($validasi['email']);
            }
            $user = User::findOrFail($id);
            $user->update($data);
            DB::commit();
            return redirect()->route('pengguna.index')->withSuccess('Data pengguna berhasil di update');
        } catch (\Exception $e) {
            DB::rollback(); //batalkan transaksi jika ada error
            return back()->withError('Kesalahan : ' . $e);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $data = User::findOrFail($id);
        $poligon = Poligon::where('id_user', $data->id)->first();
        $catatan = Catatan::where('id_user', $data->id)->first();
        $OTP = OTP::where('id_user', $data->id)->first();
        DB::beginTransaction();
        try {
    
            $defaultFoto = 'file/profile/1.jpg';
    
            // Hapus foto jika bukan default
            if ($data->foto && $data->foto !== $defaultFoto && Storage::exists($data->foto)) {
                Storage::delete($data->foto);
            }
    
            // Hapus folder milik user (misalnya: file/profile/{user_id})
            $folderProfile = "file/profile/{$data->id}";
            if (Storage::exists($folderProfile)) {
                Storage::deleteDirectory($folderProfile);
            }
            if ($OTP) {
                OTP::where('id_user', $data->id)->get()->each->delete();
            }
            if ($poligon) {
                $poligon->nama_user = $data->nama;
                $poligon->temp_id = $data->id;
                $poligon->save();
            }
            if ($catatan) {
                $catatan->nama_user = $data->nama;
                $catatan->save();
            }
            $data->delete();
            DB::commit();
            return redirect()->route('pengguna.index')->withSuccess('Data pengguna berhasil dihapus');
        } catch (\Exception $e) {
            DB::rollback(); //batalkan transaksi jika ada error
            return back()->withError('Kesalahan : ' . $e);
        }
    }

}
